// Operating system and file system utilities
/* global sc, java, Packages */

function moduleFunction(config) {
  // Initialize $$ with provided config or default
  let $$ = config || require('./debug')({ debug: 'cache', path: module.id });

  let objects = require('common/objects');

  // Java imports
  let File = global.java.io.File;
  let FileOutputStream = global.java.io.FileOutputStream;
  let FileInputStream = global.java.io.FileInputStream;
  let StringConverterUtils = global.Packages.agorum.commons.string.StringConverterUtils;
  let Charset = global.java.nio.charset.Charset;

  /**
   * Returns the directory name from a given file path.
   *
   * @param {string} path - The file path.
   * @returns {string} The directory name.
   */
  function dirName(path) {
    return path.split('/').slice(0, -1).join('/');
  }

  /**
   * Returns the base name of a given path.
   *
   * @param {string} path - The path from which to extract the base name.
   * @returns {string} The base name of the path.
   */
  function baseName(path) {
    return path.split('/').slice(-1)[0];
  }

  /**
   * Converts the content of a source object to a file and saves it to the target location.
   * @param {string} source - The source object to convert.
   * @param {string} target - The target location to save the file.
   * @returns {File} - The created file.
   */
  function contentToFile(source, target) {
    let obj = objects.find(source);
    let file = new File(target);
    if (file.exists()) file.delete();
    let fileOutputStream = new FileOutputStream(file);
    fileOutputStream.write(StringConverterUtils.toByteArray(obj.contentStream));
    obj.contentStream.close();
    fileOutputStream.close();
    return file;
  }

  /**
   * Converts a file to bytes and creates a new file object with the specified target.
   * @param {string} source - The source file path.
   * @param {string} target - The target file path.
   * @returns {object} - The newly created file object.
   */
  function fileToContent(source, target) {
    let file = new File(source);
    let fileInputStream = new FileInputStream(file);
    let obj = objects.tryFind(target);
    if (obj) obj.delete();
    obj = objects.create('file', {
      name: baseName(target),
      target: objects.find(dirName(target)),
      content: fileInputStream,
    });
    fileInputStream.close();
    return obj;
  }

  /**
   * Converts a string to a content object and returns it.
   * @param {string} string - The string to be converted to content.
   * @param {string} target - The target object's path.
   * @returns {object} - The content object created from the string.
   */
  function stringToContent(string, target) {
    let obj = objects.tryFind(target);
    if (obj) obj.delete();
    obj = objects.create('file', {
      name: baseName(target),
      target: objects.find(dirName(target)),
    });
    obj.contentString = string;
    return obj;
  }

  /**
   * Executes a process with the given options.
   * @param {Object} _opts - The options for executing the process.
   * @returns {Object} - Process execution results.
   */
  function executeProcess(_opts) {
    let opts = Object.assign(
      {
        command: null,
        args: [],
        envp: null,
        path: null,
        waitFor: true,
        timeOut: null,
        inputString: null,
        inputDelay: 10,
      },
      _opts
    );

    let ep = new global.Packages.agorum.commons.utils.ExecuteProcess();
    
    if (opts.inputString !== null) {
      ep.setInputString(opts.inputString, opts.inputDelay);
    }
    
    if (opts.timeOut !== null) {
      ep.setTimeOut(opts.timeOut);
    }
    
    let commandArray = [opts.command].concat(opts.args || []);
    $$.debug('executeProcess: command', commandArray, opts);
    ep.execute(commandArray, opts.path, opts.waitFor);
    
    let result = {
      executeProcess: ep,
      outputString: ep.getOutputString(),
      errorString: ep.getErrorString(),
      exitValue: ep.getExitValue(),
    };
    
    $$.debug('executeProcess: result', result);
    return result;
  }

  /**
   * Converts a content stream to a string.
   * @param {Object} stream - The content stream to convert.
   * @param {string} [_charset='UTF-8'] - The character set to use.
   * @returns {string} - The converted string.
   */
  function convertContentStream(stream, _charset) {
    let charset = _charset || 'UTF-8';
    let reader = new global.java.io.BufferedReader(
      new global.java.io.InputStreamReader(stream, Charset.forName(charset))
    );
    let line = null;
    let content = [];
    while ((line = reader.readLine()) !== null) {
      content.push(line);
    }
    reader.close();
    return content.join('\n');
  }

  /**
   * Writes a string to a file.
   * @param {string} string - The string to write.
   * @param {string} file - The target file path.
   * @param {string} [encoding='UTF-8'] - The encoding to use.
   */
  function writeStringToFile(string, file, encoding) {
    let inputStream = StringConverterUtils.toStream(string, encoding || 'UTF-8');
    let fileOutputStream = new FileOutputStream(file);
    StringConverterUtils.pipe(inputStream, fileOutputStream);
    fileOutputStream.close();
    inputStream.close();
  }

  // Return the module functions
  return {
    dirName: dirName,
    baseName: baseName,
    contentToFile: contentToFile,
    fileToContent: fileToContent,
    stringToContent: stringToContent,
    executeProcess: executeProcess,
    convertContentStream: convertContentStream,
    writeStringToFile: writeStringToFile
  };
}

// Initialize $$ when the module is required without parameters
let moduleExports = moduleFunction();

// Export the module function and its methods
module.exports = Object.assign(moduleFunction, moduleExports);
